// Opcja - set color 0 to XX
// - ustawia zerowy cluster na dany kolor i go nie modyfikuje podczas k-means

#pragma warning (disable: 4312)
#pragma warning (disable: 4996)

#include <math.h>
#include <vector>
#include <map>
#include <set>
#include <algorithm>
#include <string>
#include "FreeImage.h"
#include <allegro.h> 
#include "CommandLineParser.h"
#include "string_conv.h"
#include "xexfile.h"

void QuitFunc(void)
{
	allegro_exit();
	exit(0);
}

void error(char *e)
{
	allegro_message(e);
	allegro_exit();
	exit(1);
}

void error(char *e, int i)
{
	allegro_message("%s %d",e,i);
	allegro_exit();
	exit(1);
}


char *help_text=
"Quantizator v0.97 by Jakub 'Ilmenit' Debski / 01.2010\n\n"
"Usage:\n"
"Quantizator.exe InputFile [options]\n\n"
"/i=Input File\n"
"  File to process. A lot of GFX formats are supported, including JPG, PNG, GIF and BMP.\n"
"  The image is rescaled to defined width and height.\n"
"\n"
"/col=COL file\n"
"  You may use an existing COL file for quantization process.\n"
"  In this case the color selection for lines is skipped.\n"
"\n"
"/w=Output Width\n"
"  Generated file width. Must be dividable by 4."
"  Default: 160 (40 bytes in G2F), -1 to skip resizing\n"
"\n"
"/h=Output Height\n"
"  Generated file height."
"  Default: 240, -1 to skip resizing\n\n"
"/mode=Processing Modes:\n"
"  simple - basic color quantization\n"
"  local - maximizing local similarity. Slow, because all the color combinations are checked.\n"
"  dither - linear dithering\n"
"  rdither - randomized linear dithering\n"
"  cdither - chessboard linear dithering\n"
"  fdither - FloydSteinberg dithering, works only when following lines share similar palette.\n"
"  preview - different configuration combinations are called and many outputs are generated for preview.\n"
"  fastpreview - like preview but modes 'fdither' and 'local' are not included (much faster).\n"
"\n"
"/dval=Dithering Value\n"
"  0.0 to 1.0 - strength of dithering and rdithering. 0.5 is default.\n\n"
"/dist=RGB Distance Function\n"
"  approx - low cost approximation (default)\n"
"  euclid - euclidian distance\n"
"\n"
"/cf=Color Choose Function\n"
"  mc - the most common Atari color is choosen for the color (default)\n"
"  avg - averaged color is choosen for color\n"
"\n"
"/pal=Palette File\n"
"  Processing quality depends a lot on similarity between chosen Atari palette and image palette.\n"
"  The default is Palettes\\laoo.act, which covers a large color range and is close to Atari palette.\n"
"\n"
"/o=Output File\n"
"  Output file name. With this name output files COL, MIC and XEX will be generated.\n"
"  The XEX file is generated when height<=192 and width<=160\n"
"\n"
"/nsort\n"
"  Skips phase of sorting colors to speed up generation process.\n"
"  Such picture contains too many DLI changes to be usable, so use it only as preview\n"
"\n"
"/lock=RRGGBB[:RRGGBB:...]\n"
"  Locks colors into a specific RGB value for every line.\n"
"  F.e. /lock=000000:00FFFF sets COLBK to black and COLPF0 to blue.\n"
"  Locked colors are still used for the quantization process.\n"
"  Lock all the colors to have one palette on the whole picture.\n"
"\n"
"To load file into Graph2Font first load MIC file, then COL.\n"
"The screen size in bytes must be set before loading!\n\n"
"Example:\n"
"Quantizator.exe test.jpg /lock=000000 /h=192 /mode=rdither /dval=0.2 /pal=palettes\\g2f.act\n"
"Quantizator.exe test.jpg /mode=preview\n"
"\n";

void ShowHelp()
{
#if 1
	FILE *fp;
	fp=fopen("readme.txt","wt+");
	if (fp!=NULL)
	{
		fprintf(fp,"%s",help_text);
		fclose(fp);
	}
#endif
	error("Quantizator v0.97 by Jakub Debski '2010\n\nQuantizator.exe InputFile [options]");
}



int screen_color_depth;

enum e_dither_type {
	DITHER_TYPE_LINEAR, 
	DITHER_TYPE_RANDOMIZED,
	DITHER_TYPE_FLOYD,
	DITHER_TYPE_CHESSBOARD, 
};

void Message(char *message)
{
	rectfill(screen,0,400,640,420,0);
	textprintf_ex(screen, font, 0, 400, makecol(0xF0,0xF0,0xF0), 0, "%s", message);
}

void Message(char *message, int i)
{
	rectfill(screen,0,400,640,420,0);
	textprintf_ex(screen, font, 0, 400, makecol(0xF0,0xF0,0xF0), 0, "%s %d", message,i);
}


using namespace std;
using namespace Epoch::Foundation;

struct rgb {
	unsigned char b;
	unsigned char g;
	unsigned char r;
	unsigned char a;
	bool operator==(const rgb &ar)
	{
		if (r==ar.r && g==ar.g && b==ar.b)
			return true;
		else
			return false;
	}

};

// simple sort to distinguish different colors for maps, sets etc
bool operator<(const rgb &l, const rgb &r) 
{
	if (l.r<r.r && l.g<r.g && l.b<r.b)
		return true;
	else
		return false;
}

#define PIXEL2RGB(p) (*((rgb*) &p))
#define RGB2PIXEL(p) (*((int*) &p))

typedef int color_index;
typedef int pixel_number;

class screen_line {
private:
	vector < rgb > pixels;
public:
	vector < vector < pixel_number > > pixels_in_clusters; // two structures for easier search
	vector < color_index > clusters_for_pixels; // two structures for easier search
	vector < rgb > line_palette;
	void Resize(size_t i)
	{
		pixels.resize(i);
	}
	rgb& operator[](size_t i)
	{
		return pixels[i];
	}
};

// by http://www.compuphase.com/cmetric.htm
// A low-cost approximation
double RGBApproximationDistance(rgb &e1, rgb &e2)
{
	long rmean = ( (long)e1.r + (long)e2.r ) / 2;
	long r = (long)e1.r - (long)e2.r;
	long g = (long)e1.g - (long)e2.g;
	long b = (long)e1.b - (long)e2.b;
	return sqrt((double)(((512+rmean)*r*r)>>8) + 4*g*g + (((767-rmean)*b*b)>>8));
}

double RGBEuclidianDistance(rgb &col1, rgb &col2)
{
	double distance=0;
	// euclidian distance
	distance+=((double)col1.r-(double)col2.r)*((double)col1.r-(double)col2.r);
	distance+=((double)col1.g-(double)col2.g)*((double)col1.g-(double)col2.g);
	distance+=((double)col1.b-(double)col2.b)*((double)col1.b-(double)col2.b);
	return sqrt(distance);
}

inline double DotProduct (rgb &col1, rgb &col2)		// Calculate The Angle Between The 2 Vectors
{
	return col1.r * col2.r + col1.g * col2.g + col1.b * col2.b;		// Return The Angle
}

//inline double Magnitude (rgb &col)				// Calculate The Length Of The colector
//{
//	return sqrt ((double)(col.r * col.r + col.g * col.g + col.b * col.b));	// Return The Length Of The colector
//}

inline double Magnitude(rgb &col)
{
	return sqrt(DotProduct(col, col));
}


double RGBCosineDistance(rgb &col1, rgb &col2)
{
	double dotProduct = DotProduct(col1, col2);
	double magnitudeOfA = Magnitude(col1);
	double magnitudeOfB = Magnitude(col2);

	return dotProduct/(magnitudeOfA*magnitudeOfB);
}

struct Configuration {
	string input_file;
	string output_file;
	string palette_file;
	string col_file;
	string mode;
	string color_function;
	string distance_function;

	int width;
	int height;
	int destination_colors;
	double diffusion_level;
	bool sort_colors;
	vector < rgb > locked_colors; 

	CommandLineParser parser; 

	Configuration(Configuration &a)
	{
		*this=a;
	}
	Configuration(int argc, char *argv[]);
};

class Quantizator;

typedef double (*f_rgb_distance)(rgb &col1, rgb &col2);
typedef void (Quantizator::*f_choose_colors)(size_t y);

class Quantizator {
private:
	FILE *out, *in;
	PALETTE palette;
	BITMAP *input_bitmap;	
	BITMAP *expected_bitmap;
	BITMAP *output_bitmap;
	FIBITMAP *fbitmap; 

	// configuration
	Configuration cfg;

	// function pointers
	f_rgb_distance distance_function;
	f_choose_colors choose_colors_function;

	vector < screen_line > picture; 
	vector < rgb > atari_palette; // 128 colors in mode 15!

// private functions
	void ProcessLine(size_t y);  // my - using k-means clustering
	void ExtractClusters();

	// Choose color algorithms
	void ChooseColorsPerLineAverage(size_t y);
	void ChooseColorsPerLineMostCommonAtariColor(size_t y);

	void ChooseColorsForPicture(); 
	unsigned char FindAtariColor(rgb &col);

	void DiffuseError( int x, int y, double quant_error, double e_r,double e_g,double e_b);
	bool NextCombination(vector < unsigned char > & c);
	rgb  FindClosestAtariColorInLine(rgb c, size_t y);
	bool SortColors();
	bool ReassignPictureColorsToLinePalette(BITMAP *bmp);
	void InitLocalStructure();
	bool CreateBatFile(string fname, string command);

public:
	Quantizator(Configuration &c);
	bool ProcessInit();
	bool LoadInputBitmap();
	bool QuantizePicture();
	bool LocalImprovePicture();
	bool DitherPicture(e_dither_type dtype);
	bool LoadAtariPalette(const char *filename);
	bool SaveScreenData(const char *filename);
	bool SaveScreenColors(const char *filename);
	bool LoadScreenColors(const char *filename);
	bool SavePicture(string filename);
	bool SaveDLIHandler(const char *filename);
	bool GenerateXex(string filename);
	bool Preview();
};


// My k-means clustering
void Quantizator::ProcessLine(size_t y)
{
	int i,x,c;
	line(screen,0,y,input_bitmap->w,y,makecol(255,255,0));

	//////////////////////////////////////////////////////////////////////////
	// make clusters 
	vector < rgb > centroids(cfg.destination_colors);
	
	// set centroids
	// 1. The first one is the pixel with minimal distance to others pixels
	// remember about locked colors

	int best_pixel=-1;
	double distance;
	if (cfg.locked_colors.empty())
	{
		double min_distance=9999999999;
		for (i=0;i<input_bitmap->w;++i)
		{
			distance=0;
			for (x=0;x<input_bitmap->w;++x)
			{
				distance += distance_function(picture[y][x],picture[y][i]);
			}
			if (distance<min_distance)
			{
				min_distance=distance;
				best_pixel=i;
			}
		}
		centroids[0]=picture[y][best_pixel];
	}

	for (i=0;i<cfg.locked_colors.size();++i)
		centroids[i]=cfg.locked_colors[i];

	// 2. The next clusters are selected as pixels with the max. distance to previously choosen centroids
	for (i=cfg.locked_colors.size();i<cfg.destination_colors;++i)
	{
		best_pixel=-1;
		double max_distance=0;
		for (x=0;x<input_bitmap->w;++x)
		{
			distance=0;
			for (c=0;c<i;++c)
			{
				distance += distance_function(picture[y][x],centroids[c]);
			}
			if (distance>=max_distance)
			{
				max_distance=distance;
				best_pixel=x;
			}
		}
		centroids[i]=picture[y][best_pixel];
	}

	// assign pixels to clusters
	fill(picture[y].clusters_for_pixels.begin(),picture[y].clusters_for_pixels.end(),0);

	int best_cluster;
	bool assign_changed;
	do 
	{
		assign_changed=FALSE;
		best_cluster=0;
//		best_cluster=-1;
		// for each pixel find the closest center and assign pixel to it
		for (x=0;x<input_bitmap->w;++x)
		{
			double min_distance=9999999999;
			for (i=0;i<cfg.destination_colors;++i)
			{
				double distance = distance_function(picture[y][x],centroids[i]);
				if (distance<min_distance)
				{
					best_cluster=i;
					min_distance=distance;
				}
			}
			if (picture[y].clusters_for_pixels[x]!=best_cluster)
			{
				picture[y].clusters_for_pixels[x]=best_cluster;
				assign_changed=TRUE;
			}
		}
		// move centroids to new positions - average value 
		for (i=cfg.locked_colors.size();i<cfg.destination_colors;++i)
		{
			int count=0;
			int r,g,b;
			r=g=b=0;
			for (x=0;x<input_bitmap->w;++x)
			{
				if (picture[y].clusters_for_pixels[x]==i)
				{
					r+=picture[y][x].r;
					g+=picture[y][x].g;
					b+=picture[y][x].b;
					++count;
				}
			}
			if (count>0)
			{
				centroids[i].r=r/count;
				centroids[i].g=g/count;
				centroids[i].b=b/count;
			}
		}
	} while (assign_changed);

	//////////////////////////////////////////////////////////////////////////
	// Store result
	pixel_number cluster;
	picture[y].pixels_in_clusters.resize(cfg.destination_colors);
	for (x=0;x<input_bitmap->w;++x)
	{
		cluster=picture[y].clusters_for_pixels[x];
		picture[y].pixels_in_clusters[cluster].push_back(x);
	}
	blit(input_bitmap,screen,0,y,0,y,input_bitmap->w,1);
}

void Quantizator::ChooseColorsPerLineMostCommonAtariColor(size_t y)
{
	size_t i;
	// We have the pixels and clusters
	// Calculate the average / median color for each cluster
	size_t current_cluster;
	map <rgb, size_t> colors;

	for (current_cluster=0;current_cluster<picture[y].pixels_in_clusters.size();++current_cluster)
	{
		colors.clear();
		for (i=0;i<picture[y].pixels_in_clusters[current_cluster].size();++i)
		{
			size_t pixel=picture[y].pixels_in_clusters[current_cluster][i];
			rgb atari_color = atari_palette[FindAtariColor(picture[y][pixel])];
			colors[atari_color]++;
		}
		rgb temp;
		if (colors.size()>0)
		{
			map <rgb, size_t>::iterator i,_i,j;
			size_t max=0;
			for (i=colors.begin(),_i=colors.end();i!=_i;++i)
			{
				if ((*i).second>max)
				{
					max=(*i).second;
					j=i;
				}
			}
			temp = (*j).first;
		}
		else
		{
			temp.r=0x0;
			temp.g=0x0;
			temp.b=0x0;
		}
		picture[y].line_palette[current_cluster]=temp;
	}
}

void Quantizator::ChooseColorsPerLineAverage(size_t y)
{
	size_t i;
	// We have the pixels and clusters
	// Calculate the average / median color for each cluster
	double r_av,g_av,b_av;

	size_t current_cluster;
	for (current_cluster=0;current_cluster<picture[y].pixels_in_clusters.size();++current_cluster)
	{
		if (current_cluster<cfg.locked_colors.size())
		{
			picture[y].line_palette[current_cluster]=cfg.locked_colors[current_cluster];
		}
		else
		{
			r_av=g_av=b_av=0;
			size_t pixel;
			for (i=0;i<picture[y].pixels_in_clusters[current_cluster].size();++i)
			{
				pixel=picture[y].pixels_in_clusters[current_cluster][i];
				r_av+=picture[y][pixel].r;
				g_av+=picture[y][pixel].g;
				b_av+=picture[y][pixel].b;
			}
			rgb temp;

			if (picture[y].pixels_in_clusters[current_cluster].size()!=0)
			{
				temp.r=(unsigned char) (r_av/picture[y].pixels_in_clusters[current_cluster].size());
				temp.g=(unsigned char) (g_av/picture[y].pixels_in_clusters[current_cluster].size());
				temp.b=(unsigned char) (b_av/picture[y].pixels_in_clusters[current_cluster].size());
			}
			else
			{

				// average color in line !!! - for example for dithering
				temp.r=0x0;
				temp.g=0x0;
				temp.b=0x0;
			}
			picture[y].line_palette[current_cluster]=temp;
		}
	}
}

void Quantizator::ChooseColorsForPicture()
{
	int y;
	Message("Choosing best colors for lines");
	for (y=0;y<input_bitmap->h;++y)
	{
		(this->*choose_colors_function)(y);
	}
}


void Quantizator::ExtractClusters()
{
	int y;

	Message("Analyzing line colors");
	for (y=0;y<input_bitmap->h;++y)
	{
		ProcessLine(y);
	}
}

void Quantizator::DiffuseError( int x, int y, double quant_error, double e_r,double e_g,double e_b)
{
	rgb p;
	int r,g,b;
	int pixel = getpixel(output_bitmap,x,y);
	p = PIXEL2RGB(pixel);
	r = p.r;
	g = p.g;
	b = p.b;
	r += e_r * quant_error;
	g += e_g * quant_error;
	b += e_b * quant_error;
	if (r>255)
		r=255;
	if (r<0)
		r=0;
	if (g>255)
		g=255;
	if (g<0)
		g=0;
	if (b>255)
		g=255;
	if (g<0)
		g=0;
	p.r=r;
	p.g=g;
	p.b=b;
	pixel = RGB2PIXEL(p);
	putpixel(output_bitmap,x,y,pixel);
}

rgb Quantizator::FindClosestAtariColorInLine(rgb c, size_t y)
{
	rgb closest;
	double min_distance=999999999;
	double distance;
	for (size_t i=0;i<cfg.destination_colors;++i)
	{
		rgb atari_color=atari_palette[FindAtariColor(picture[y].line_palette[i])];
		distance=distance_function(c,atari_color);
		if (distance<min_distance)
		{
			closest=atari_color;
			min_distance=distance;
		}
	}
	return closest;
}

bool Quantizator::DitherPicture(e_dither_type dtype)
{
	int x,y;
	Message("Dithering picture");

	// copy input_bitmap to output
	blit(input_bitmap,output_bitmap,0,0,0,0,input_bitmap->w,input_bitmap->h);

	// Draw new picture on the screen
	for (y=0;y<output_bitmap->h;++y)
	{
		for (x=0;x<output_bitmap->w;++x)
		{
			// oldpixel := pixel[x][y]

				rgb oldpixel;
				int pixel = getpixel(output_bitmap,x,y);
				oldpixel = PIXEL2RGB(pixel);

			// newpixel := find_closest_palette_color(oldpixel)
#if 1
				// Atari Colors limited to DLI
				rgb newpixel;
				newpixel = FindClosestAtariColorInLine(oldpixel,y);
#else
				// Atari Colors not limited to DLI
				rgb newpixel=atari_palette[FindAtariColor(oldpixel)];
#endif
				pixel = RGB2PIXEL(newpixel);
				// pixel[x][y] := newpixel
				putpixel(output_bitmap,x,y,pixel);
				putpixel(screen,x+output_bitmap->w,y,pixel);

			// quant_error := oldpixel - newpixel

				double qe_r, qe_g, qe_b;

				if (dtype==DITHER_TYPE_RANDOMIZED)
				{
					qe_r=qe_g=qe_b=rand()%128;
				}
				else
				{
					qe_r = (double)oldpixel.r - (double) newpixel.r;
					qe_g = (double)oldpixel.g - (double) newpixel.g;
					qe_b = (double)oldpixel.b - (double) newpixel.b;
				}
				/* Standard FloydSteinberg uses 4 pixels to diffuse
				pixel[x+1][y] := pixel[x+1][y] + 7/16 * quant_error
				pixel[x-1][y+1] := pixel[x-1][y+1] + 3/16 * quant_error
				pixel[x][y+1] := pixel[x][y+1] + 5/16 * quant_error
				pixel[x+1][y+1] := pixel[x+1][y+1] + 1/16 * quant_error
				*/
				if (dtype==DITHER_TYPE_FLOYD)
				{
					DiffuseError( x+1, y,   7.0/16.0, qe_r,qe_g,qe_b);
					DiffuseError( x-1, y+1, 3.0/16.0, qe_r,qe_g,qe_b);
					DiffuseError( x  , y+1, 5.0/16.0, qe_r,qe_g,qe_b);
					DiffuseError( x+1, y+1, 1.0/16.0, qe_r,qe_g,qe_b);
				}
				else if  (dtype==DITHER_TYPE_LINEAR)
				{
					// We use a single line diffusion, because of different palettes in lines
					// The diffusion is randomized to avoid vertical lines artifacts
					DiffuseError( x+1, y,   ((double) rand())/((double)RAND_MAX + 1)*cfg.diffusion_level, qe_r,qe_g,qe_b);
				}
				else // chessboard dithering
				{
					if ((x+y)%2==0)
						DiffuseError( x+1, y, cfg.diffusion_level, qe_r,qe_g,qe_b);
				}
		}
	}
	return true;
}

bool Quantizator::SavePicture(string filename)
{
	Message("Saving picture");

	stretch_blit(output_bitmap,expected_bitmap,0,0,input_bitmap->w,input_bitmap->h,0,0,expected_bitmap->w,expected_bitmap->h);
	// copy expected bitmap to FreeImage bitmap to save it as png
//	save_bitmap("test.bmp",expected_bitmap,palette);
//	FIBITMAP *f_outbitmap = FreeImage_Allocate(expected_bitmap->w,expected_bitmap->h, screen_color_depth);	
	FIBITMAP *f_outbitmap = FreeImage_Allocate(expected_bitmap->w,expected_bitmap->h, 24);	

	size_t x,y;
	int color;
	for (y=0;y<expected_bitmap->h;++y)
	{
		for (x=0;x<expected_bitmap->w;++x)
		{
			color = getpixel( expected_bitmap,x,y);
			FreeImage_SetPixelColor(f_outbitmap, x, y, (RGBQUAD *)&color);
		}
	}

	FreeImage_FlipVertical(f_outbitmap);
	FreeImage_Save(FIF_PNG,f_outbitmap,filename.c_str());
	return true;
}

bool Quantizator::GenerateXex(string filename)
{
	size_t x,y;
	unsigned char *mic_data=xexfile+0x1C20; // in the XEX file
	unsigned char *dli_data=xexfile+0x13C; // in the XEX file
	unsigned char *dli_start=dli_data;
	unsigned char a,b,c,d;
	bool possible_error;

	Message("Generating XEX file");

	memset(mic_data,0,160*192/4);

	// copy mic_data
	if (output_bitmap->h<=192 && output_bitmap->w<=160)
	{
		// copy picture data
		memset(mic_data,0,output_bitmap->w*192 / 4); // 4 colors in one byte
		for(y=0;y<output_bitmap->h;++y)
		{
			// encode 4 pixel colors in byte
			for (x=0;x<(output_bitmap->w>160?output_bitmap->w:160);x+=4)
			{
				unsigned char pix=0;
				if (x<output_bitmap->w)
				{
					a=picture[y].clusters_for_pixels[x];
					b=picture[y].clusters_for_pixels[x+1];
					c=picture[y].clusters_for_pixels[x+2];
					d=picture[y].clusters_for_pixels[x+3];
				}
				else
				{
					a=b=c=d=0;
				}
				pix |= a<<6;
				pix |= b<<4;
				pix |= c<<2;
				pix |= d;
				*mic_data++=pix;
			}
		}
		// Create DLI code
		unsigned char reg;

		int last_a_register=-1;
		vector <int> previous_colors(cfg.destination_colors,-1);

		for(y=0;y<output_bitmap->h;++y)
		{
			// wsync
			*dli_data++=0x8D;
			*dli_data++=0x0A;
			*dli_data++=0xD4;

			for (size_t i=0;i<cfg.destination_colors;++i)
			{
				unsigned char atari_color = FindAtariColor(picture[y].line_palette[i])*2;
				switch(i)
				{
					case 0:
						reg=0x1A;
					break;
					case 1:
						reg=0x16;
						break;
					case 2:
						reg=0x17;
						break;
					case 3:
						reg=0x18;
						break;
				}
				if (last_a_register!=atari_color)
				{
					last_a_register=atari_color;

					*dli_data++=0xA9; // LDA
					*dli_data++=atari_color; // value
				}

				if (previous_colors[i]!=atari_color)
				{
					*dli_data++=0x8D; // STA
					*dli_data++=reg; // reg
					*dli_data++=0xD0; // reg

					previous_colors[i]=atari_color;
				}
			}
		}
		// 3 - wsync + 5*4colors - (lda/sta) = 23*192 + 20
		// store black border color at the end
		*dli_data++=0xA9; // LDA
		*dli_data++=0; // 0
		*dli_data++=0x8D; // STA
		*dli_data++=0x1A; // border
		*dli_data++=0xD0; // reg
		*dli_data=0x40; // RTI
//*/
		size_t 	count = dli_data-dli_start;

		FILE *fp=fopen(string(filename+".xex").c_str(),"wb+");
		if (!fp)
			error("Error saving XEX file");
		fwrite(xexfile,1,XEXFILE_LEN,fp);
		fclose(fp);
	}
	return true;
}

bool Quantizator::SaveDLIHandler(const char *filename)
{
	Message("Saving DLI Handler");

	// Convert picture color info to DL handler
	int y;
	int last_a_register=-1;

	vector <int> previous_colors(cfg.destination_colors,-1);

	FILE *fp=fopen(filename,"wt+");
	if (!fp)
		error("Error saving DLI handler");
	for(y=0;y<input_bitmap->h;++y)
	{
		fprintf(fp," sta $D40A ; wsync %d\n",y);
		for (size_t i=0;i<cfg.destination_colors;++i)
		{
			rgb cluster_color=picture[y].line_palette[i];
			unsigned char col=FindAtariColor(cluster_color)*2; // we use 128 colors in palette

			if (previous_colors[i]!=col)
			{
				previous_colors[i]=col;
				if (col!=last_a_register)
				{
					fprintf(fp," lda #$%02X\n",col);
					last_a_register=col;
				}
				switch (i)
				{
				case 0:
					fprintf(fp," sta $D01A\n");
					break;
				case 1:
					fprintf(fp," sta $D016\n");
					break;
				case 2:
					fprintf(fp," sta $D017\n");
					break;
				case 3:
					fprintf(fp," sta $D018\n");
					break;
				}
			}
		}
	}
	fprintf(fp," sta $D40A ; wsync\n");
	fprintf(fp," lda #$0\n");
	fprintf(fp," sta $D01A\n");
	fclose(fp);
	return true;
}

bool Quantizator::QuantizePicture()
{
	int x,y;
	Message("Quantizing picture");

	// Draw new picture on the screen
	for (y=0;y<input_bitmap->h;++y)
	{
		for (x=0;x<input_bitmap->w;++x)
		{
			size_t cluster=picture[y].clusters_for_pixels[x];
			rgb cluster_color=picture[y].line_palette[cluster];

/* Without Atari colors
			int color=RGB2PIXEL(cluster_color);
*/
			rgb atari_color=atari_palette[FindAtariColor(cluster_color)];
			int color=RGB2PIXEL(atari_color);

			putpixel(screen,x+output_bitmap->w,y,color);
			putpixel(output_bitmap,x,y,color);
		}
	}
	return true;
}

bool Quantizator::NextCombination(vector < unsigned char > & c)
{
	size_t pos=-1;
increase_next:
	++pos;
	if (pos>=c.size())
		return false;
	++c[pos];
	if (c[pos]>=cfg.destination_colors)
	{
		// zero all the previous
		for (size_t i=0;i<=pos;++i)
			c[pos]=0;
		goto increase_next;
	}
	return true;
}

bool Quantizator::LocalImprovePicture()
{
	int x,y,a,b;
	rgb lold[2][2];

	Message("Maximizing local similarity");

	vector < unsigned char > combinations;
	vector < unsigned char > best_combination;
	vector < rgb > colors;
	vector < rgb > best_colors;
	combinations.resize(4);
	best_combination.resize(4);
	colors.resize(4);
	best_colors.resize(4);

	// Draw new picture on the screen
	for (y=0;y<input_bitmap->h-1;y+=2)
	{
		for (x=0;x<input_bitmap->w-1;x+=2)
		{
			// get local pixels
			for (b=0;b<2;++b)
			{
				for (a=0;a<2;++a)
				{
					int pixel=getpixel(input_bitmap,x+a,y+b);
					lold[b][a]=PIXEL2RGB(pixel);
				}
			}

			// Test all the combinations of colors to minimize the local distance
			fill(combinations.begin(),combinations.end(),0);
		
			size_t pos=0;

			// increase combination

			double min_distance=99999999999;
			for(;;)
			{
				// x,y
				double distance=0;
				colors[0]=atari_palette[FindAtariColor(picture[y].line_palette[combinations[0]])];
				distance+=distance_function(colors[0],lold[0][0]);
				// x+1,y
				colors[1]=atari_palette[FindAtariColor(picture[y].line_palette[combinations[1]])];
				distance+=distance_function(colors[1],lold[0][1]);
				// x,y+1
				colors[2]=atari_palette[FindAtariColor(picture[y+1].line_palette[combinations[2]])];
				distance+=distance_function(colors[2],lold[1][0]);
				// x+1,y+1
				colors[3]=atari_palette[FindAtariColor(picture[y+1].line_palette[combinations[3]])];
				distance+=distance_function(colors[3],lold[1][1]);
				if (distance<min_distance)
				{
					min_distance=distance;
					best_combination=combinations;
					best_colors=colors;
				}
				// increase color combination
				if (!NextCombination(combinations))
					break;
			}
			// Set colors by the best local combination
			
			putpixel(screen,x  +output_bitmap->w,y  ,RGB2PIXEL(best_colors[0]));
			putpixel(screen,x+1+output_bitmap->w,y  ,RGB2PIXEL(best_colors[1]));
			putpixel(screen,x  +output_bitmap->w,y+1,RGB2PIXEL(best_colors[2]));
			putpixel(screen,x+1+output_bitmap->w,y+1,RGB2PIXEL(best_colors[3]));

			putpixel(output_bitmap,x  ,y  ,RGB2PIXEL(best_colors[0]));
			putpixel(output_bitmap,x+1,y  ,RGB2PIXEL(best_colors[1]));
			putpixel(output_bitmap,x  ,y+1,RGB2PIXEL(best_colors[2]));
			putpixel(output_bitmap,x+1,y+1,RGB2PIXEL(best_colors[3]));

		}
	}
	return true;
}

bool Quantizator::LoadInputBitmap()
{
	Message("Loading and initializing file");
	fbitmap = FreeImage_Load(FreeImage_GetFileType(cfg.input_file.c_str()), cfg.input_file.c_str(), 0);
	if (!fbitmap)
		error("Error loading input file");
	if (cfg.width<=0 || cfg.height<=0)
	{
		cfg.width = FreeImage_GetWidth(fbitmap);
		cfg.height = FreeImage_GetHeight(fbitmap);
	}
	else
		fbitmap=FreeImage_Rescale(fbitmap,cfg.width,cfg.height,FILTER_LANCZOS3);

	if (screen_color_depth==32)
		fbitmap=FreeImage_ConvertTo32Bits(fbitmap);
	else
		fbitmap=FreeImage_ConvertTo24Bits(fbitmap);

	FreeImage_FlipVertical(fbitmap);

	set_palette(palette);
	input_bitmap  = create_bitmap_ex(screen_color_depth,cfg.width,cfg.height);
	output_bitmap  = create_bitmap_ex(screen_color_depth,cfg.width,cfg.height);
	expected_bitmap = create_bitmap_ex(screen_color_depth,cfg.width*2,cfg.height);
	return true;
}

void Quantizator::InitLocalStructure()
{
	size_t x,y;

	//////////////////////////////////////////////////////////////////////////
	// Set our structure size
	picture.resize(input_bitmap->h);

	// Copy data to input_bitmap and to our structure
	RGBQUAD fpixel;
	for (y=0;y<input_bitmap->h;++y)
	{
		for (x=0;x<input_bitmap->w;++x)
		{
			picture[y].Resize(input_bitmap->w);
			FreeImage_GetPixelColor(fbitmap, x, y, &fpixel);
			putpixel( input_bitmap,x,y,makecol(fpixel.rgbRed,fpixel.rgbGreen,fpixel.rgbBlue));
			picture[y][x]=PIXEL2RGB(fpixel);
			picture[y].line_palette.resize(cfg.destination_colors);
			picture[y].clusters_for_pixels.resize(input_bitmap->w);
		}
	}
	clear_bitmap(screen);
	// Show our picture
	draw_sprite(screen, input_bitmap, 0, 0);
};

bool Quantizator::CreateBatFile(string fname, string command)
{
	FILE *fp=fopen(fname.c_str(),"wt+");
	fprintf(fp,"%s",command.c_str());
	fclose(fp);
	return true;
}

bool Quantizator::Preview()
{
	const int no_distance_functions=2;
	const int no_choose_color_functions=2;

	char *choose_color_params[no_distance_functions]=
	{
		" /cf=avg",
		"" // ignore default parameter " /cf=mostcommon"
	};

	f_choose_colors choose_color_functions[no_distance_functions]=
	{
		&Quantizator::ChooseColorsPerLineAverage,
		&Quantizator::ChooseColorsPerLineMostCommonAtariColor
	};

	char *distance_functions_params[no_distance_functions]=
	{
		" /dist=euclid",
		"", // ignore default parameter " /dist=approx"
	};

	f_rgb_distance distance_functions[no_distance_functions]=
	{
		RGBEuclidianDistance,
		RGBApproximationDistance
	};

	size_t d;
	size_t c;
	int config_number=1;
	string cfg_name;
	for (d=0;d<no_distance_functions;++d)
	{

		choose_colors_function=choose_color_functions[d];

		for (c=0;c<no_choose_color_functions;++c)
		{
			InitLocalStructure();

			distance_function=distance_functions[c];

			ExtractClusters();
			ChooseColorsForPicture();

			string local_command_line=string("Quantizator.exe ") + cfg.input_file;
			local_command_line+=choose_color_params[c];
			local_command_line+=distance_functions_params[c];

			if (cfg.parser.getValue("w","160")!="160")
				local_command_line+=" /w="+cfg.parser.getValue("w","160");

			if (cfg.parser.getValue("h","240")!="240")
				local_command_line+=" /h="+cfg.parser.getValue("h","240");

			if (cfg.parser.getValue("dval","0.5")!="0.5")
				local_command_line+=" /dval="+cfg.parser.getValue("dval","0.5");

			if (cfg.parser.getValue("lock","")!="")
				local_command_line+=" /lock="+cfg.parser.getValue("lock","");

			if (cfg.palette_file!="Palettes\\laoo.act")
				local_command_line+=" /pal="+cfg.palette_file;

			string command_line;
			command_line=local_command_line;
			QuantizePicture();
			cfg_name=string("configuration") + Value2String<int>(config_number++);
			CreateBatFile(cfg_name+".bat",command_line);
			ReassignPictureColorsToLinePalette(output_bitmap);
			SavePicture (cfg_name+".png");
			rectfill(screen,input_bitmap->w,0,input_bitmap->w*2,input_bitmap->h,0);

			DitherPicture(DITHER_TYPE_LINEAR);
			command_line=local_command_line+" /mode=dither";
			cfg_name=string("configuration") + Value2String<int>(config_number++);
			CreateBatFile(cfg_name+".bat",command_line);
			ReassignPictureColorsToLinePalette(output_bitmap);
			SavePicture (cfg_name+".png");
			rectfill(screen,input_bitmap->w,0,input_bitmap->w*2,input_bitmap->h,0);

			DitherPicture(DITHER_TYPE_RANDOMIZED);
			command_line=local_command_line+" /mode=rdither";
			cfg_name=string("configuration") + Value2String<int>(config_number++);
			CreateBatFile(cfg_name+".bat",command_line);
			ReassignPictureColorsToLinePalette(output_bitmap);
			SavePicture (cfg_name+".png");
			rectfill(screen,input_bitmap->w,0,input_bitmap->w*2,input_bitmap->h,0);

			DitherPicture(DITHER_TYPE_CHESSBOARD);
			command_line=local_command_line+" /mode=cdither";
			cfg_name=string("configuration") + Value2String<int>(config_number++);
			CreateBatFile(cfg_name+".bat",command_line);
			ReassignPictureColorsToLinePalette(output_bitmap);
			SavePicture (cfg_name+".png");
			rectfill(screen,input_bitmap->w,0,input_bitmap->w*2,input_bitmap->h,0);

			if (cfg.mode=="preview")
			{
				DitherPicture(DITHER_TYPE_FLOYD);
				command_line=local_command_line+" /mode=fdither";
				cfg_name=string("configuration") + Value2String<int>(config_number++);
				CreateBatFile(cfg_name+".bat",command_line);
				ReassignPictureColorsToLinePalette(output_bitmap);
				SavePicture (cfg_name+".png");
				rectfill(screen,input_bitmap->w,0,input_bitmap->w*2,input_bitmap->h,0);

				command_line=local_command_line+" /mode=local";
				LocalImprovePicture();
				cfg_name=string("configuration") + Value2String<int>(config_number++);
				CreateBatFile(cfg_name+".bat",command_line);
				ReassignPictureColorsToLinePalette(output_bitmap);
				SavePicture (cfg_name+".png");
				rectfill(screen,input_bitmap->w,0,input_bitmap->w*2,input_bitmap->h,0);
			}
		}
	}
	return true;
}

bool Quantizator::ProcessInit()
{
	InitLocalStructure();

	if (cfg.color_function=="avg")
		choose_colors_function=&Quantizator::ChooseColorsPerLineAverage;
	else
		choose_colors_function=&Quantizator::ChooseColorsPerLineMostCommonAtariColor;

	if (cfg.distance_function=="euclid")
		distance_function=RGBEuclidianDistance;
	else
		distance_function=RGBApproximationDistance;

	if (cfg.col_file!="")
	{
		LoadScreenColors(cfg.col_file.c_str());
	}
	else if (cfg.mode!="wu")
	{
		ExtractClusters();
		ChooseColorsForPicture();
		if (cfg.sort_colors)
			SortColors();
	}
	ReassignPictureColorsToLinePalette(input_bitmap);
	return true;
}

bool Quantizator::LoadAtariPalette(const char *filename)
{
	Message("Loading palette");
	size_t i;
	rgb col;
	col.a=0;
	FILE *fp=fopen(filename,"rb");
	if (!fp)
		error("Error opening .act palette file");
	for (i=0;i<256;++i)
	{
		col.r=fgetc(fp);
		col.g=fgetc(fp);
		col.b=fgetc(fp);
		// limit it to 128 colors!
		// use every second color
		if (i%2==0)
			atari_palette.push_back(col);
	}
	fclose(fp);
	return true;
}

inline unsigned char Quantizator::FindAtariColor(rgb &col)
{
	size_t i;
	// Find the most similar color in the Atari Palette
	unsigned char most_similar=0;
	double distance;
	double min_distance=99999999999;
	for(i=0;i<128;++i)
	{
		distance=distance_function(col,atari_palette[i]);
		if (distance<min_distance)
		{
			min_distance=distance;
			most_similar=i;
		}
	}
	return most_similar;
}

bool Quantizator::LoadScreenColors(const char *filename)
{
	int y;
	Message("Loading screen colors");

	FILE *fp=fopen(filename,"rb");
	if (!fp)
		error("Error opening COL file");

	for (size_t i=0;i<cfg.destination_colors;++i)
	{
		for(y=0;y<256;++y)
		{
			unsigned char val=fgetc(fp);
			if (feof(fp))
			{
				val=i;
			}
				
			if (y<input_bitmap->h)
			{
				picture[y].line_palette[i]=atari_palette[val/2];
			}
		}
	}
	fclose(fp);
	return true;
}

bool Quantizator::SaveScreenColors(const char *filename)
{
	int y;
	Message("Saving screen colors");

	FILE *fp=fopen(filename,"wb+");
	if (!fp)
		error("Error saving COL screen colors");
	for (size_t i=0;i<cfg.destination_colors;++i)
	{
		for(y=0;y<256;++y)
		{
			if (y<input_bitmap->h)
			{
				fputc(FindAtariColor(picture[y].line_palette[i])*2,fp);
			}
			else
				fputc(0,fp);
		}
	}
	fclose(fp);
	return true;
}

bool Quantizator::ReassignPictureColorsToLinePalette(BITMAP *bmp)
{
	int x,y;
	Message("Reassigning colors");

	// reassign data to the current output_bitmap
	for(y=0;y<bmp->h;++y)
	{
		for (x=0;x<bmp->w;++x)
		{			
			int pixel= getpixel(bmp,x,y);
			rgb color = PIXEL2RGB(pixel);
			double distance;
			double min_distance=9999999999;
			for (size_t i=0;i<cfg.destination_colors;++i)
			{
				distance = distance_function(color,atari_palette[FindAtariColor(picture[y].line_palette[i])]);
				if (distance<min_distance)
				{
					picture[y].clusters_for_pixels[x]=i;
					min_distance=distance;
				}
			}
		}
	}
	return true;
}


bool Quantizator::SortColors()
{
	int y,i,a;
	Message("Sorting colors");

	// Sort colors, so similar will be on the same index
	for(y=0;y<output_bitmap->h;++y)
	{
		putpixel(screen,499,y,makecol(0,255,0));
		putpixel(screen,500+cfg.destination_colors*20+1,y,makecol(255,255,0));
		for (size_t i=0;i<cfg.destination_colors;++i)
		{
			line(screen,500+i*20,y,520+i*20,y,RGB2PIXEL(atari_palette[FindAtariColor(picture[y].line_palette[i])]));
		}
	}

	// process next lines
	for(y=1;y<output_bitmap->h;++y)
	{
		double distance;

		// Swap colors in the line palette 

		vector <double> min_distance_s(cfg.destination_colors,9999999999);

		for (int i=cfg.locked_colors.size();i<cfg.destination_colors;++i)
		{
			double min_distance=999999999;
			int closest;
			for (int j=cfg.locked_colors.size();j<cfg.destination_colors;++j)
			{
				rgb prev=atari_palette[FindAtariColor(picture[y-1].line_palette[i])];
				rgb curr=atari_palette[FindAtariColor(picture[y].line_palette[j])];

				distance=distance_function(prev,curr);
				if (distance<min_distance)
				{
					min_distance=distance;					
					closest=j;
				}
			}
			if (min_distance<min_distance_s[i])
			{
				min_distance_s[i]=min_distance;
				swap(picture[y].line_palette[i],picture[y].line_palette[closest]);
			}
		}
		for(a=0;a<output_bitmap->h;++a)
		{
			putpixel(screen,399,y,makecol(255,255,0));
			putpixel(screen,400+cfg.destination_colors*20+1,y,makecol(255,255,0));
			for (size_t i=0;i<cfg.destination_colors;++i)
			{
				line(screen,400+i*20,y,420+i*20,y,RGB2PIXEL(atari_palette[FindAtariColor(picture[y].line_palette[i])]));
			}
		}
	}
	// Move darkest color to the 0 position (background)
	vector <double> brightness(cfg.destination_colors,0);
	for(y=0;y<output_bitmap->h;++y)
	{		
		for (i=0;i<cfg.destination_colors;++i)
		{
			rgb c = picture[y].line_palette[i];
			double luma=(0.299 * c.r) + (0.587 * c.g) + (0.114 * c.b);
			brightness[i]+=luma;
		}
	}
	// Find darkest color column
	double min_bright=999999999;
	int darkest;
	for (i=0;i<cfg.destination_colors;++i)
	{
		if (brightness[i]<min_bright)
		{
			min_bright=brightness[i];
			darkest=i;
		}
	}
	if (darkest!=0)
	{
		for(y=0;y<output_bitmap->h;++y)
		{
			swap(picture[y].line_palette[0],picture[y].line_palette[darkest]);
		}
	}
	for(y=0;y<output_bitmap->h;++y)
	{
		for (size_t i=0;i<cfg.destination_colors;++i)
		{
			line(screen,400+i*20,y,420+i*20,y,RGB2PIXEL(atari_palette[FindAtariColor(picture[y].line_palette[i])]));
		}
	}

	ReassignPictureColorsToLinePalette(input_bitmap);
	return true;
}

bool Quantizator::SaveScreenData(const char *filename)
{

	int x,y,a,b,c,d;
	FILE *fp=fopen(filename,"wb+");
	if (!fp)
		error("Error saving MIC screen data");

	Message("Saving screen data");
	for(y=0;y<output_bitmap->h;++y)
	{
		// encode 4 pixel colors in byte
		for (x=0;x<output_bitmap->w;x+=4)
		{
			unsigned char pix=0;
			a=picture[y].clusters_for_pixels[x];
			b=picture[y].clusters_for_pixels[x+1];
			c=picture[y].clusters_for_pixels[x+2];
			d=picture[y].clusters_for_pixels[x+3];
			pix |= a<<6;
			pix |= b<<4;
			pix |= c<<2;
			pix |= d;
			fwrite(&pix,1,1,fp);
		}
	}
	fclose(fp);
	return true;
}


Quantizator::Quantizator(Configuration &a_c)
:
cfg(a_c)
{
}

Configuration::Configuration(int argc, char *argv[])
{
	parser.parse(argc, argv);

	input_file = parser.getValue("i","NoFileName");
	output_file = parser.getValue("o","output.png");
	mode = parser.getValue("mode","simple");
	palette_file = parser.getValue("pal","Palettes\\laoo.act");
	col_file = parser.getValue("col","");
	mode = parser.getValue("mode","simple");

	if (parser.switchExists("palette"))
		palette_file = parser.getValue("palette","Palettes\\laoo.act");

	if (parser.switchExists("preview"))
		mode="preview";
	if (parser.switchExists("fastpreview"))
		mode="fastpreview";

	if (!parser.switchExists("i"))
	{
		string temp;
		if (argc>1)
		{
			temp=argv[1];
			if (temp.find("/")==string::npos)
				input_file=temp;
		}
		else
			ShowHelp();
	}
	if (parser.switchExists("help") || parser.nonInterpretedExists("--help") || parser.switchExists("?"))
		ShowHelp();

	if (parser.switchExists("sprites"))
		destination_colors=5;
	else 
		destination_colors=4;

	string width_value = parser.getValue("w","160");
	width=String2Value<int>(width_value);
	if (width%4!=0)
		error("Width must be dividable by 4, because 4 pixels are encoded in byte.");

	string height_value = parser.getValue("h","240");
	height=String2Value<int>(height_value);

	string dither_value = parser.getValue("dval","0.5");
	diffusion_level=String2Value<double>(dither_value);

	sort_colors=!(parser.switchExists("nsort"));

	string lock = parser.getValue("lock","");
	// parse locked colors
	rgb color;
	while (lock.size()>=6)
	{
		color.r=String2HexValue<int>(lock.substr(0,2));
		color.g=String2HexValue<int>(lock.substr(2,2));
		color.b=String2HexValue<int>(lock.substr(3,2));
		locked_colors.push_back(color);
		if (lock.size()>7)
			lock=lock.substr(7);
		else
			break;
	}

	color_function = parser.getValue("cf","mc");
	if (color_function=="average")
		color_function = "avg";
	distance_function = parser.getValue("dist","approx");
}

int main(int argc, char *argv[])
{
	//////////////////////////////////////////////////////////////////////////
	allegro_init(); // Initialize Allegro
	set_close_button_callback(QuitFunc);
	FreeImage_Initialise(TRUE);
	screen_color_depth = desktop_color_depth();
	set_color_depth(screen_color_depth);
	set_gfx_mode(GFX_AUTODETECT_WINDOWED, 640,480,0,0); // Change our graphics mode to 640x480
	set_display_switch_mode(SWITCH_BACKGROUND);

	Configuration cfg(argc, argv);

	Quantizator q(cfg);

	q.LoadAtariPalette(cfg.palette_file.c_str());
	q.LoadInputBitmap();

	if (cfg.mode=="preview" || cfg.mode=="fastpreview")
		q.Preview();
	else
	{
		q.ProcessInit();
		if (cfg.mode=="simple")
			q.QuantizePicture();
		else if (cfg.mode=="local")
			q.LocalImprovePicture();
		else if (cfg.mode=="dither")
			q.DitherPicture(DITHER_TYPE_LINEAR);
		else if (cfg.mode=="rdither")
			q.DitherPicture(DITHER_TYPE_RANDOMIZED);
		else if (cfg.mode=="fdither")
			q.DitherPicture(DITHER_TYPE_FLOYD);
		else if (cfg.mode=="cdither")
			q.DitherPicture(DITHER_TYPE_CHESSBOARD);
		else
			error("Processing mode unknown");	
	

		q.SaveScreenData  (string(cfg.output_file+".mic").c_str());
		q.SaveScreenColors(string(cfg.output_file+".col").c_str());
		q.SavePicture     (cfg.output_file);
		q.SaveDLIHandler  (string(cfg.output_file+".s").c_str());
		q.GenerateXex  (cfg.output_file);
	}
	// FreeImage_Unload(fbitmap); - leak if not freed, commented for /mode=preview

	return 0; // Exit with no errors
}

END_OF_MAIN() // This must be called right after the closing bracket of your MAIN function.
// It is Allegro specific.